/**   sauditor - sample auditor  
  *   Copyright (C) 2009  James Shuttleworth
  *
  *   Contact: james@dis-dot-dat.net
  *
  *   This program is free software; you can redistribute it and/or modify
  *   it under the terms of the GNU General Public License as published by
  *   the Free Software Foundation; either version 2 of the License, or
  *   (at your option) any later version.
  *
  *   This program is distributed in the hope that it will be useful,
  *   but WITHOUT ANY WARRANTY; without even the implied warranty of
  *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  *   GNU General Public License for more details.
  *
  *   You should have received a copy of the GNU General Public License
  *   along with this program; if not, write to the Free Software
  *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  *
  */
#include <X11/Xlib.h>
#include <getopt.h>
#include <iostream>
#include <fstream>
#include <unistd.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/param.h>
#include <string.h>

#include <sndfile.h>
#include <samplerate.h>
#include <curses.h>
#include "dirFollow.h"
#include "jackPart.h"
#include <menu.h>    
#include "fftpart.h"
#include <fftw3.h>
#include <glib.h>
#include <stdlib.h>

//#include <pthread.h>
#define FILE_SEPARATOR_STR "/"
using namespace std;

#include "version.h"

void createSA(long len, double* data);
void showSA();
void hideSA();
void toggleSA();
void dispVersions();
void dispHelp();
void dispInfo(const char* inf);
char* cat(const char* a, const char* b);
string getFormatString(int);
string getSubFormatString(int);
int file_select(const dirent*);
int checkMask(int, int);
void playSound(SNDFILE*,SF_INFO*);
int playFile(const char* file);
int changeDir(const char* dir);
//rabitcode quality
int quality= SRC_SINC_BEST_QUALITY;    
extern  int alphasort();
string curpath;    
//typedef jack_default_audio_sample_t sample_t;    
bool drawspec=false;
void listSetup(const char* path);
void listSetup();
#define NIL (0)       // A name for the void pointer
//Show spectrum analyser?
bool doSA=false;  

//Has the selection changed?
bool changed=true;
int fcount;
int dcount;
dirFollow dir;

int screenx,screeny;
typedef int(*gofuncptr)(const char*);

ITEM** fileitems;
MENU *filemenu;
ITEM *cur_item;
WINDOW *fileswin;
WINDOW *spec;


Display *dpy;
Window sawin;
GC sagc;

void createSA(long len, double* data){
  static long lastfftlength=0;
  static int bins=10;
  static bool fftopened=false;  
  if(doSA){ 

    int blackColor = BlackPixel(dpy, DefaultScreen(dpy));
    int whiteColor = WhitePixel(dpy, DefaultScreen(dpy));  

    //XClearWindow(dpy,sawin);
    if(!fftopened || lastfftlength!=len){
      fftpart_setup(len,1,len/2);
      fftopened=true;
      lastfftlength=len; 
      //cerr << "\nfftpart_setup called.";     
    }
    int w,h;
    w=200;
    h=100;
    double* vals=(double*)malloc(sizeof(double)*(w));  
    fftpart_grab(w-2,data,vals); 
    for(int i=0;i<w-2;i++){
      int num=int(vals[i]*h);
      XSetForeground(dpy, sagc, blackColor);
      XDrawLine(dpy, sawin, sagc, i, 0, i, h);      
      XSetForeground(dpy, sagc, whiteColor);
      XDrawLine(dpy, sawin, sagc, i, h, i, h-num);      
    }
    free(vals);
    XFlush(dpy);
    //dispInfo("SA!");
  }
  else
    return;
}



char* trimpad(char* in, int width){
  char* result=new char[width+1];
  if(strlen(in)>width){
    for(int i=0;i<width;i++)
      result[i]=in[i];
  }else{
    for(int i=0;i<strlen(in);i++)
      result[i]=in[i];
    for(int i=strlen(in);i<width;i++)
      result[i]=' ';
  }
  result[width]='\0';
  return result;
}


void listSetup(){
  //cerr<<"[1]Calling listSetup with: "<<curpath.c_str()<<endl;
  listSetup(cat(curpath.c_str(),""));
}


void dispHelp(){
  dispInfo(" ");
  dispInfo("Commandline options:");
  dispInfo("\t-h Help message");
  dispInfo("\t-v Show version info and exit");
  dispInfo("\t-a Show all files (don't exclude 'dot' files)");
  dispInfo("\t-f Show X spectrum analyser window");
  dispInfo(" ");
  dispInfo("Keys:");
  dispInfo("\tENTER\tPlay file under cursor");
  dispInfo("\tp\tPlay file under cursor");
  dispInfo("\tSPACE\tToggle playing");
  dispInfo("\tf\tToggle SA window");
  dispInfo("\tr\tRefresh screen");
  dispInfo("\tv\tVersion Info");
  dispInfo("\th\tHelp");
  dispInfo("\tUse cursor keys, page up/down, home end for navigation");
  dispInfo(" ");
}

void dispVersions(){
  char buffer [128];
  dispInfo(cat("sauditor: ",version));
  sf_command (NULL, SFC_GET_LIB_VERSION, buffer, sizeof (buffer)) ;
  dispInfo(cat("libsndfile: ",buffer));
  dispInfo(cat("(n)curses: ",curses_version()));
  dispInfo(cat("JACK:",jackversion));
  dispInfo(cat("SRC:",srcversion));
  dispInfo(cat("FFTW:",fftwversion));
  
  int tsr=jackpart_samplerate();
  sprintf(buffer, "%i", tsr);
  dispInfo(cat("Jack server running at ",cat(buffer,"Hz")));
}

void dispInfo(const char* inf){
  static int pos=1;
  int w,h;
  getmaxyx(spec, h, w);
  if(strlen(inf)==0)
    pos=0;
  if(drawspec){
    if(pos>=h-2){
      pos=1;
      wclear(spec);
    }
    wrefresh(spec);
    //wclear(spec);
    wmove(spec, pos, 1);    
    waddstr(spec,inf);
    box(spec,0,0);
    wrefresh(spec);
    pos++;
  }  
}

void myrefresh(){

  //clear();
  wclear(fileswin);

  //box(stdscr, 0, 0);
  refresh();
  listSetup();
  wrefresh(fileswin);
  box(fileswin, 0, 0);
  drawspec=false;
  int w,h;
  getmaxyx(stdscr, h, w);
  if(!drawspec){
    spec=newwin(h-2, w/2-2, 1, w/2+1);
    drawspec=true;
  }
  dispInfo("");
}






void listSetup(const char* path){
  curpath=string(path);

  getmaxyx(stdscr, screeny, screenx) ;
  if(!fileswin)
    fileswin = newwin(10,10,1,1);
  if(filemenu){
    unpost_menu(filemenu);    
    free_menu(filemenu);
    for(int i=0;i<fcount+dcount;i++){
      //cerr<<"freeing: "<<fileitems[i]->name.str<<"["<<fileitems[i]->description.str<<"]"<<endl;
      free_item(fileitems[i]);
    }
  }

  wresize(fileswin,screeny-2, screenx/2);
  string** files;
  string** dirs;
  //cerr<<"got: "<<path<<endl;

  dir.setPath(path);
  fcount = dir.getFiles(files);

  dcount = dir.getDirs(dirs);
  

  fileitems=(ITEM **)calloc(dcount+fcount + 1 /*for null*/ , sizeof(ITEM *));
  for(int i=0;i<dcount;i++){
    fileitems[i] = new_item(trimpad(cat("->",g_path_get_basename(dirs[i]->c_str())),screenx/2-5), dirs[i]->c_str());
    fileitems[i]->userptr=(void*)&changeDir;
   
    //cerr<<fileitems[i]->name.str<<":"<<fileitems[i]->description.str<<":"<<*dirs[i]<<endl;
    
  }

  for(int i=0;i<fcount;i++){
    fileitems[i+dcount] = new_item(trimpad(g_path_get_basename(files[i]->c_str()),screenx/2-5), files[i]->c_str());
    //----------------------------------------------
    //Patch from David <david@justroots.com>
    //Fixes crash on NFS mounts, apparently.  
    if(fileitems[i+dcount])
      fileitems[i+dcount]->userptr=(void*)&playFile;
    //replaces: fileitems[i+dcount]->userptr=(void*)&playFile;
    //----------------------------------------------
  }
  //cerr<<dcount<<" dirs and "<<fcount<<" files\n";
  fileitems[fcount+dcount] = (ITEM *)NULL;
  filemenu = new_menu((ITEM **)fileitems);

  menu_opts_off(filemenu, O_SHOWDESC);  



  keypad(fileswin, TRUE); 

  //nodelay(fileswin, TRUE);
  
  static WINDOW* fsubwin=derwin(fileswin, 10, 10, 1, 1);
  wresize(fsubwin,screeny-4,screenx/2-2);
  set_menu_format(filemenu, screeny-4, 1);
  set_menu_win(filemenu, fileswin);
  set_menu_sub(filemenu, fsubwin);
  post_menu(filemenu);

  if(fcount>0)
    free(files);
  if(dcount>0)
    free(dirs);

}

void showSA(){
  if(!dpy){
    dispInfo(" Initialising spectrym analyser window.");
    dpy = XOpenDisplay(NIL);
    int blackColor = BlackPixel(dpy, DefaultScreen(dpy));
    int whiteColor = WhitePixel(dpy, DefaultScreen(dpy));
    
    // Create the window
    
    sawin = XCreateSimpleWindow(dpy, DefaultRootWindow(dpy), 0, 0, 200, 100, 0, blackColor, blackColor);
    XSelectInput(dpy, sawin, StructureNotifyMask);
    XMapWindow(dpy, sawin);
    sagc = XCreateGC(dpy, sawin, 0, NIL);
    XSetForeground(dpy, sagc, whiteColor);
    for(;;) {
      XEvent e;
      XNextEvent(dpy, &e);
      if (e.type == MapNotify)
	break;
    }
  }
  doSA=true;
}

void hideSA(){
  dispInfo(" Removing spectrum analyser window.");
  XCloseDisplay(dpy);
  dpy=NIL;
  doSA=false;
}

void toggleSA(){
  if(doSA){
    hideSA();
  }else{
    showSA();
  }
}

int main(int argc, char* argv[]){
  char* path=cat(getcwd(NULL,0),"/");

  doSA=false;
  int opt;
  while((opt=getopt(argc,argv,"fvah"))!=-1){
    if(opt=='h'){
      cout << "\nUseage:\n\t"<<argv[0]<<" [OPTIONS]\n\nOptions are:\n\t-h This help message\n\t-v Show version info and exit\n\t-a Show all files\n\t-f Show X spectrum analyser window\n\n";
      cout<<"Keys:\n\tENTER\tPlay file under cursor\n\tp\tPlay file under cursor\n\tSPACE\tToggle playing\n\tf\tToggle SA window\n\tr\tRefresh screen\n\tv\tVersion Info\n\th\tHelp\n\tUse cursor keys, page up/down, home end for navigation\n\n";
      opt='v';
    }
      
    if(opt=='v'){
      cout<<"\nSauditor version "<<version<<endl<<"Created by James Shuttleworth <james@dis-dot-dat.net>\nReleased under the Gnu GPL\n\n";
      return 0;
    }

    if(opt=='f')
      toggleSA();
    if(opt=='a')
      dir.withDots=true;
  }


    

  //addstr("Browsing ");
  //addstr(path);
  //addstr("\n");
  cout<<"";


  if(!jackpart_open()){
    cerr<<"\nProblem initialising JACK";
    return 1;
  }

  jackpart_setproc(&createSA);

  initscr();
  //nodelay(stdscr, TRUE);
  noecho();
  //keypad(stdscr, TRUE);

  //cerr<<"[2]Calling listSetup with: "<<path<<endl;
  listSetup(path);  

  myrefresh();
  dispInfo("");

  dispVersions();


  dispInfo("Press H for help.");
  dispInfo(" ");

  if(doSA){
    doSA=false;
    toggleSA();
  }


  int c;
  while((c = wgetch(fileswin)) != 'q'&c!='Q'){

    switch(c){
    case 'r':
    case 'R':
      //drawspec=false;
      myrefresh();
      break;
    case KEY_RESIZE:
      myrefresh();
      //cerr<<"Refreshing...\n";
      
      break;
    case 'p':
    case 'P':
    case KEY_ENTER:
    case '\r':
    case '\n':
      cur_item=current_item(filemenu);
      ////cerr<<"Called on item "<<cur_item->name.str<<"["<<cur_item->description.str<<"]\n";
      ((gofuncptr)(cur_item->userptr))(cur_item->description.str);
      break;

    case ' ':
      if(jackpart_playing())
	jackpart_stop();
      else{
	cur_item=current_item(filemenu);
	gofuncptr f=(gofuncptr)(cur_item->userptr);
	f(cur_item->description.str);
      }
      break;
    case KEY_NPAGE:

      menu_driver(filemenu, REQ_SCR_DPAGE);
      changed=true;

      break;
    case KEY_PPAGE:

      menu_driver(filemenu, REQ_SCR_UPAGE);
      changed=true;

      break;
      
    case KEY_DOWN:

      menu_driver(filemenu, REQ_DOWN_ITEM);
      changed=true;

      break;
    case KEY_UP:

      menu_driver(filemenu, REQ_UP_ITEM);
      changed=true;
      break;

    case KEY_END:
      menu_driver(filemenu, REQ_LAST_ITEM);
      changed=true;
      break;

    case KEY_HOME:
      menu_driver(filemenu, REQ_FIRST_ITEM);
      changed=true;
      break;

    case 'f':
    case 'F':
      toggleSA();
      break;

    case 'v':
    case 'V':
      dispVersions();
      break;

    case 'h':
    case 'H':
      dispHelp();
      break;

    }

  }
  free(path);
  unpost_menu(filemenu);    
  free_menu(filemenu);
  for(int i=0;i<fcount+dcount+1;i++)
    free_item(fileitems[i]);
  endwin();
  jackpart_stop();
  jackpart_close();
  fftpart_close();
  cerr << "fftpart_close() called.\n";
  return 0;
}

int changeDir(const char* ndir){
  if(strcmp(g_path_get_basename(ndir),".")==0)
    ;  //Do nout
    //cerr<<"No change\n";
  else if(strcmp(g_path_get_basename(ndir),"..")==0){
    //cerr<<"Need to go back "<<g_path_get_dirname(g_path_get_dirname(ndir))<<endl;
    listSetup(cat(g_path_get_dirname(g_path_get_dirname(ndir)),FILE_SEPARATOR_STR));
  }else{
    //cerr <<"Changing to "<<ndir<<endl;
    listSetup(cat(ndir,FILE_SEPARATOR_STR));
  }
  
}

int playFile(const char* file){
  jackpart_stop();
  if(changed){
    SF_INFO* info=(SF_INFO*)malloc(sizeof(SF_INFO));
    info->format=0;
    SNDFILE* f=sf_open(file,SFM_READ,info);

    if((info->format & SF_FORMAT_TYPEMASK) != 0){
      dispInfo("Loading audio.");
      playSound(f,info);
      dispInfo("Playing audio.");
    }else{
      dispInfo("Failed to load.  Probably an unsupported type.");
      return false;
    }
    free(info);
    sf_close(f);
    changed=false;
  }else{
    dispInfo("Playing audio.");
    jackpart_start();
  }
  return true;
}
    
int file_select(const struct dirent* entry){
  if ((strcmp(entry->d_name, ".") == 0) || (strcmp(entry->d_name, "..") == 0))
    return 0;
  else
    return 1;
}
    
string getFormatString(int formatNum){
  string aString="Unknown";
  int newnum=formatNum & SF_FORMAT_TYPEMASK;
  if( newnum != 0){
    string wav="SUPPORTED";
    return wav;
  }
  return aString;
}
    
string getSubFormatString(int formatNum){
  string aString("");
  int newnum=formatNum & SF_FORMAT_SUBMASK;
    
  if( newnum== SF_FORMAT_PCM_S8)
    aString.append("8 bit, signed");    
    
  if( newnum== SF_FORMAT_PCM_16)
    aString.append("16 bit, signed");
    
  if( newnum == SF_FORMAT_PCM_24)
    aString.append("24 bit, signed");
    
    
  if( newnum == SF_FORMAT_PCM_32)
    aString.append("32 bit, signed");
    
    
  if( newnum == SF_FORMAT_PCM_U8)
    aString.append("8 bit, unsigned");
    
    
    
  return aString;
}
    
char* cat(const char* a, const char* b){
  char* result=new char[strlen(a)+strlen(b)+1];
  for(int i=0;i<strlen(a);i++)
    result[i]=a[i];
  for(int i=0;i<strlen(b);i++)
    result[i+strlen(a)]=b[i];
  result[strlen(a)+strlen(b)]='\0';
  return result;
}
    
void playSound(SNDFILE* f,SF_INFO* info){
  long frames=info->frames*info->channels;
  //addstr("PLaying "<< frames<<" frames\n";
  long samples=frames;
  sample_t* sound=(sample_t *) malloc (samples * sizeof(sample_t));
  sf_read_float(f,sound,samples);
  //combine channels into mono
  if(info->channels>1){
    long newframes=info->frames;
    sample_t* newsound=(sample_t *) malloc (newframes * sizeof(sample_t));
    for(int i=0;i<newframes;i++){
      sample_t tmp=0;
      for(int j=0;j<info->channels;j++){
    	tmp+=sound[i*info->channels+j];
      }
      tmp/=info->channels;
      newsound[i]=tmp;
    }
    free(sound);
    sound=(sample_t *) malloc (newframes * sizeof(sample_t));
    for(int i=0;i<newframes;i++)
      sound[i]=newsound[i];
    free(newsound);
    samples=newframes;
    
  }
  //Rabitize!
  SRC_DATA *data=(SRC_DATA*)malloc(samples*sizeof(SRC_DATA));
  data->data_in=sound;
  data->input_frames=samples;
  data->src_ratio=(float)jackpart_samplerate()/info->samplerate;
  //cout << data->src_ratio;
  int newframes=(int)(samples*data->src_ratio);
  data->output_frames=newframes;
  data->data_out=(float*)malloc(newframes*sizeof(float));
  src_simple(data,quality,1);
  sound=data->data_out;
  free(data->data_in);
  free(data);
  samples=newframes;
  jackpart_stop();
  jackpart_setSound(sound,samples);
  jackpart_start();
  free(sound);

}
