/**   This file is part of:
  *   sauditor - sample auditor  
  *   Copyright (C) 2009  James Shuttleworth
  *
  *   Contact: james@dis-dot-dat.net
  *
  *   This program is free software; you can redistribute it and/or modify
  *   it under the terms of the GNU General Public License as published by
  *   the Free Software Foundation; either version 2 of the License, or
  *   (at your option) any later version.
  *
  *   This program is distributed in the hope that it will be useful,
  *   but WITHOUT ANY WARRANTY; without even the implied warranty of
  *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  *   GNU General Public License for more details.
  *
  *   You should have received a copy of the GNU General Public License
  *   along with this program; if not, write to the Free Software
  *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
  *
  */

#include "jackPart.h"
#include <iostream>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>

using namespace std;

void(*proc_callback)(long,double*);
jack_client_t* jp_client;
jack_port_t* jp_output_port;

jack_nframes_t jp_sr;
sample_t* jp_sound;
jack_nframes_t jp_offset;
bool jp_play;
jack_nframes_t jp_samples;

pthread_mutex_t jp_mutex = PTHREAD_MUTEX_INITIALIZER;


bool jp_started=false;



int jp_process (jack_nframes_t nframes, void *arg){
  pthread_mutex_lock(&jp_mutex);
  
  /*grab our output buffer*/
  sample_t *out = (sample_t *) jack_port_get_buffer (jp_output_port, nframes);


  if(proc_callback){
    //clone
    double* v=(double*)malloc(sizeof(double)*nframes);
    for(int i=0;i<nframes;i++)
      v[i]=out[i];
    proc_callback(nframes,v);
    free(v);
  }
  /*For each required sample*/
  for(jack_nframes_t i=0;i<nframes;i++){
    /*Copy the sample at the current position in the sound to the buffer*/
    /*Should use memcpy or somesuch, really*/
    if(jp_play){
      if(jp_offset>=jp_samples){
	jp_play=false;
	jp_offset=0;
      }else{
	out[i]=jp_sound[jp_offset];
	jp_offset++;
      }
    }else{
      out[i]=0;
    }
  }


  pthread_mutex_unlock(&jp_mutex);
  return 0;
}


int jp_srate (jack_nframes_t nframes, void *arg){
  jp_sr=nframes;
  return 0;
}

void jp_error (const char *desc){
  cerr<< "JACK error: "<<  desc<<endl;
  jackpart_close();
}

void jp_jack_shutdown (void *arg){
  //exit (1);
}


void jackpart_close(){
  if(jp_sound)
    free(jp_sound);
  if(jp_client)
    jack_client_close (jp_client);
}

int jackpart_open(){
  if(jp_started)
    return false;

  jp_offset=0;
  jp_play=false;

  const char **ports;
  //cerr<<"Setting error f"<<endl;
  jack_set_error_function (jp_error);
  //cerr<<"Creating client"<<endl;
  if ((jp_client = jack_client_new ("sauditor")) == 0) {
    cerr<< "jack server not running?\n";
    jackpart_close();
    return false;
  }
  //cerr<<"Setting callback"<<endl;
  jack_set_process_callback (jp_client, jp_process, 0);
  //cerr<<"setting samplerate f"<<endl;
  jack_set_sample_rate_callback (jp_client, jp_srate, 0);
  //cerr<<"setting shutdown f"<<endl;
  jack_on_shutdown (jp_client, jp_jack_shutdown, 0);
  
  
  //cerr<<"getting samplerate"<<endl;
  jp_sr=jack_get_sample_rate (jp_client);
  //cerr<<"creating output port"<<endl;
  jp_output_port = jack_port_register (jp_client, "output", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
  if (jack_activate (jp_client)) {
    cerr<<"cannot activate client\n";
    jackpart_close();
    return false;
  }
  
  //cerr<<"getting port list"<<endl;
  /* connect the ports*/
  if ((ports = jack_get_ports (jp_client, NULL, NULL, JackPortIsPhysical|JackPortIsInput)) == NULL) {
    cerr<<"Cannot find any physical playback ports\n";
    jackpart_close();
    return false;
  }
  //cerr<<"Connecting to ports"<<endl;
  int i=0;
  while(ports[i]!=NULL){
      if (jack_connect (jp_client, jack_port_name (jp_output_port), ports[i])) {
	cerr<< "cannot connect output ports\n";
	jackpart_close();
	return false;
      }
      i++;
  }
  
  free (ports);
  jp_started=true;
  return true;
}

void jackpart_setSound(const sample_t* snd, long nsamples){
  pthread_mutex_lock(&jp_mutex);
  if(jp_sound)
    free(jp_sound);  
  jp_sound=(sample_t *) malloc (nsamples * sizeof(sample_t));
  memcpy(jp_sound,snd,nsamples*sizeof(sample_t));
  jp_samples=nsamples;
  pthread_mutex_unlock(&jp_mutex);
}

void jackpart_start(){
  pthread_mutex_lock(&jp_mutex);
  jp_play=true;
  jp_offset=0;
  pthread_mutex_unlock(&jp_mutex);
}
    
void jackpart_stop(){
  pthread_mutex_lock(&jp_mutex);
  jp_offset=0;
  jp_play=false;    
  pthread_mutex_unlock(&jp_mutex);
}

int jackpart_samplerate(){
  return jp_sr;
}

bool jackpart_playing(){
  bool answer;
  pthread_mutex_lock(&jp_mutex);
  answer=jp_play;
  pthread_mutex_unlock(&jp_mutex);
  return jp_play;
}



void jackpart_setproc(void(*callback)(long,double*)){
  proc_callback=callback;
}
